import numpy as np
import heapq
import math

class Node:
    def __init__(self, x, y, t, speed, cost=0, parent=None):
        self.x = x
        self.y = y
        self.t = t
        self.speed = speed
        self.cost = cost
        self.parent = parent
        self.g = np.inf # cost from start to node
        self.h = np.inf # heuristic cost from node to goal
        self.f = np.inf # total cost of node

    def __lt__(self, other):
        return self.cost < other.cost

class AStarPlanner:
    def __init__(self, grid_fn, preference_cost_fn, start, goal, speed, resolution):
        self.grid_fn = grid_fn
        self.grid_shape = grid_fn(0).shape
        self.preference_cost_fn = preference_cost_fn
        self.start = start
        self.goal = goal
        self.speed = speed
        self.resolution = resolution
        self.open_list = []
        self.closed_list = []
        self.path = []

    def heuristic(self, node1, node2):
        return math.sqrt((node1.x - node2.x)**2 + (node1.y - node2.y)**2)

    def distance(self, node1, node2):
        return math.sqrt((node1.x - node2.x)**2 + (node1.y - node2.y)**2)

    def get_neighbors(self, node):
        neighbors = []
        for i in range(-1, 2):
            for j in range(-1, 2):
                if i == 0 and j == 0:
                    continue
                x = node.x + i
                y = node.y + j
                t = node.t + 1
                if x >= 0 and x < self.grid_shape[0] and y >= 0 and y < self.grid_shape[1]:
                    if self.grid_fn(t)[x, y] == 0:
                        neighbors.append(Node(x, y, t, self.speed, parent=node))
        return neighbors

    def reconstruct_path(self, current):
        path = []
        while current is not None:
            path.append((current.x, current.y, current.t))
            current = current.parent
        return path[::-1]

    def plan(self, start, goal):
        open_heap = []
        closed_set = set()
        start.g = 0
        start.h = self.heuristic(start, goal)
        start.f = start.g + start.h
        heapq.heappush(open_heap, (start.f, start))

        while open_heap:
            current = heapq.heappop(open_heap)[1]
            if (current.x, current.y) == (goal.x, goal.y):
                return self.reconstruct_path(current)
            closed_set.add((current.x, current.y, current.t))
            for neighbor in self.get_neighbors(current):
                if (neighbor.x, neighbor.y, neighbor.t) in closed_set:
                    continue
                skip = False
                neighbor.parent = current
                neighbor.g = current.g + self.distance(current, neighbor) + self.preference_cost_fn(neighbor)
                neighbor.h = self.heuristic(neighbor, goal)
                neighbor.f = neighbor.g + neighbor.h
                for f, open_node in open_heap:
                    if (neighbor.x, neighbor.y, neighbor.t) == (open_node.x, open_node.y, open_node.t) and neighbor.f >= open_node.f:
                        skip = True
                        continue
                if not skip:
                    heapq.heappush(open_heap, (neighbor.f, neighbor))

if __name__ == "__main__":
    grid = np.array([[0, 0, 0, 0, 0],
                     [0, 1, 1, 1, 0],
                     [0, 1, 0, 1, 0],
                     [0, 1, 0, 1, 0],
                     [0, 0, 0, 0, 0]])
    start = Node(0, 0, 0, 1)
    goal = Node(3, 2, 0, 1)
    astar = AStarPlanner(grid, start, goal, 1, 1)
    path = astar.plan(start, goal)
    print(path)
